<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Lead;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class LandingController extends Controller
{
    /**
     * Show the landing page.
     */
    public function index()
    {
        return view('landing');
    }

    /**
     * Handle Lead Capture via API.
     */
    public function storeLead(Request $request)
    {
        // 1. Validate
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'company_name' => 'nullable|string|max:255',
            'website' => 'nullable|url',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        // 2. Score Calculation
        $score = $this->calculateScore($request);

        // 3. Store
        $lead = Lead::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
            'company_name' => $request->company_name,
            'website' => $request->website,
            'source' => $request->source ?? 'Landing',
            'lead_score' => $score,
            'status' => 'New',
        ]);

        // 4. Trigger Automation (Placeholder for now)
        // AutomationService::newLead($lead);

        return response()->json([
            'status' => 'success',
            'message' => 'Lead capture successful',
            'lead_id' => $lead->id
        ], 201);
    }

    /**
     * Calculate initial lead score.
     */
    private function calculateScore(Request $request)
    {
        $score = 0;

        // No website provided or weak
        if (!$request->website) {
            $score += 30;
        }

        // Business Type logic (if provided)
        if ($request->business_type && in_array(strtolower($request->business_type), ['store', 'clinic', 'restaurant'])) {
            $score += 20;
        }

        // Has Email (Validated above)
        $score += 10;

        // Valid WhatsApp (Simple check length)
        if (Str::length($request->phone) > 9) {
            $score += 10;
        }

        return $score;
    }
}
